using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class implements partial success update verify code 
	/// response message. 
	/// </summary>
	public sealed class SecurityUpdateVerifyCodeDivisionRequiredResponseMessage : SecurityBasePartialSuccessResponseMessage
	{
		/// <summary>
		/// VistALink message response type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string ResponseType = "AV.UpdateVC";

		private DivisionInfoCollection _divisions;

		/// <summary>
		/// This constructor allows setting only list of divisions 
		/// user has to select from. There are must be at least 2 divisions in the collection. 
		/// </summary>
		/// <param name="divisions">
		///		List of divisions that user must select from (divisions user has access to). 
		///		There must be at least 2 divisions in the collection.
		///	</param>
		public SecurityUpdateVerifyCodeDivisionRequiredResponseMessage( DivisionInfoCollection divisions ) 
			: this( divisions, null ) {}

		/// <summary>
		/// This constructor allows setting list of divisions 
		/// user has to select from and optional informative message. 
		/// There are must be at least 2 divisions in the collection. 
		/// </summary>
		/// <param name="divisions">
		///		List of divisions that user must select from (divisions user has access to). 
		///		There must be at least 2 divisions in the collection. Required parameter.
		///	</param>
		///	<param name="message">Optional informative message. Expected to be human-readable.</param>
		public SecurityUpdateVerifyCodeDivisionRequiredResponseMessage( DivisionInfoCollection divisions, string message )
			: base( ResponseType, message ) 
		{
			if( divisions == null )
				throw( new ArgumentNullException( "divisions" ) );

			if( divisions.Count < 2 ) 
				throw( new ArgumentOutOfRangeException( "divisions", SR.Exceptions.DivisionListAtLeastTwoEntriesRequired() ) );

			_divisions = divisions;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public SecurityUpdateVerifyCodeDivisionRequiredResponseMessage( XmlDocument sourceDoc )
			: base( ResponseType, sourceDoc ) 
		{
			_divisions = DivisionInfoCollection.Parse( XmlParseGetPartialSuccessElement( sourceDoc ) );

			if( _divisions.Count < 2 )
				throw( new XmlParseException( SR.Exceptions.DivisionListAtLeastTwoEntriesRequired() ) );
		}

		/// <summary>
		/// Overriden member addind divisions information to the partial success element
		/// created by the base class. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlPartialSuccessNodeContent( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			base.WriteXmlPartialSuccessNodeContent( writer );

			_divisions.WriteDivisionsListToXml( writer );
		}

		/// <summary>
		/// List of divisions available for user to select from.
		/// </summary>
		public DivisionInfoCollection Divisions
		{
			get
			{
				return _divisions;
			}
		}
	}
}
